/**************************************************************************************

Copyright (c) Hilscher Gesellschaft fuer Systemautomation mbH. All Rights Reserved.

***************************************************************************************

  $Id: $:

  Description:
    Demo application of the CIFX Toolkit "Low-Level" Hardware Access Functions

  Changes:
    Date        Description
    -----------------------------------------------------------------------------------
    2019-04-05  Use Hilscher Definitions
                Change configuration handling. Remove protocol specific configuration handling
                (consult protocol specific examples and manuals)

    2018-08-09  fixed pclint warnings

    2018-07-26  Changed memset()/memcpy() to OS_ functions
                Changed memory area initialization to OS_Memset()

    2014-08-27  Added HWIF function support, checked with CIFX_TOOLKIT_HWIF option setting

    2013-05-21  Missing to change OS_Memcopy to HWIF_WRITEN in IODemo
                - handle output

    2013-01-28  Updated sources to be usable for OS and nonOS examples
                - Converted examples from cpp sources to c sources
                - Moved OS specific example functions
                  (e.g. cifXTkHWFunctions_GetDPMPointer()) to separate
                  c-module (cifXHWFunctionsOS.c)

    2012-09-06  Replaced DPM access (OS_Memcpy()) by the corresponding
                hardware read/write macro (HWIF_WRITEN/HWIF_READN)
                - Removed OS dependant code.

    2012-08-16  Added example for indication handling

    2012-05-03  Added a configuration sample for POWERLINK

    2012-03-19  cifXTkHWFunctions_InitializeDataStructures extended
                by checking for DPM is available including a timeout handling

    2012-03-05  Added a sleep of 100ms before starting with device handling.
                Necessary becaus firmware needs a momnt to set the actual COS state

    2011-08-16  - Set Board_NO to 0 to use the first available board
                - main() function restructured
                - fixing parameter oder in IODemo,  DEV_WaitForBitState() for
                  the input image

    2011-03-17  initial version

**************************************************************************************/

#include "OS_Includes.h"
#include "cifXHWFunctions.h"
#include "cifXEndianess.h"

/* Error & Type Def. */
#include "CIFXErrors.h"
#include "Hil_Results.h" /* packet results */

#ifdef CIFX_TOOLKIT_HWIF
  #include "SerialDPMInterface.h"
#endif

/* general device functions */
#include "DEV_InterfaceUser.h"
#include "DEV_FileUser.h"

#include "Fieldbus_Functions.h"

int  cifXTkHWFunctions_GetDPMPointer( uint8_t** ppbDPM, uint32_t* pulDPMSize);
void cifXTkHWFunctions_FreeDPMPointer( uint8_t* pbDPM);

/* ============================================================================= */
/* Global information and definitions                                            */
/* ============================================================================= */

/* global trace level referenced by the hardware function module (cifXHWFunctions.c) */
uint32_t                  g_ulTraceLevel        = 0;

/* My source ID (packet communication)         */
uint32_t                  ulSourceID            = 0x43214321;

/* Cards protocol and communication class      */
uint8_t                   bHandleIndications    = 0;
uint16_t                  usProtocollClass      = 0;
uint16_t                  usCommunicationClass  = 0;

/* com channel number where firmware is placed */
#define                   COM_CHANNEL           0     /* Use the first Channel on the choosen board */

/* cycles to run IO_DEMO()                     */
#define                   DEMO_CYCLES           10000
/* size of IO transfer data (bytes)            */
#define                   SIZE_OF_IODATA        64

/* device instance channel count               */
#define                   DEV_CHANNEL_COUNT     COM_CHANNEL + 1
#define                   COM_CHANNEL_OFFS      2


/* Return values of main */
#define DEV_NO_ERROR           0
/* DPM identification error                             */
#define ERR_DEV_DPM_IDENT      1
/* DEV_Init() failed!                                   */
#define ERR_DEV_INIT           2
/* DEV_Ready() request system channel failed!           */
#define ERR_DEV_SYS_READY      3
/* DEV_Ready() request communication channel failed!    */
#define ERR_DEV_COM_READY      4
/* Sending Warmstart packet failed!                     */
#define ERR_DEV_WARMSTART      5
/* DEV_Communicating() request failed!                  */
#define ERR_DEV_COM            6
/* DEV_Demo() failed!                                   */
#define ERR_DEV_DEMO           7
/* DPM mapping failed!                                  */
#define ERR_DEV_MAP            8
/* No configuration found!                              */
#define ERR_DEV_CONFIG         9
/* DPM Layout unknown                                   */
#define ERR_DEV_DPM_LAYOUT     10
/* Firmware missing                                     */
#define ERR_DEV_FW_MISSING     11



/*****************************************************************************/
/*! User trace function
*   right while cifXTKitAddDevice is being processed
*   \param ptDevInstance  Device instance
*   \param ulTraceLevel   Trace level
*   \param szFormat       Format string                                      */
/*****************************************************************************/
void USER_Trace(PDEVICEINSTANCE ptDevInstance, uint32_t ulTraceLevel, const char* szFormat, ...)
{
  UNREFERENCED_PARAMETER(ulTraceLevel);
  UNREFERENCED_PARAMETER(ptDevInstance);
  UNREFERENCED_PARAMETER(szFormat);
}


/*****************************************************************************/
/*! Identify protocol and create configuration ('warmstart') packet
 * TODO: Build configuration packet
 *
 * \param ptChannel        channel instance
 * \return CIFX_NO_ERROR on success                                          */
/*****************************************************************************/
int32_t ProcessDeviceConfiguration(  CHANNELINSTANCE* ptChannel)
{
  int32_t         lRet         = CIFX_NO_ERROR;

  CIFX_PACKET     tSendPacket;    /* Create a default packet buffer */

  /* Set pointer to the device instance structure and the system channel */
  PDEVICEINSTANCE           ptDevInst       = (PDEVICEINSTANCE)ptChannel->pvDeviceInstance;
  HIL_DPM_SYSTEM_CHANNEL_T* ptSystemChannel = (HIL_DPM_SYSTEM_CHANNEL_T*)ptDevInst->tSystemDevice.pbDPMChannelStart;

  OS_Memset( &tSendPacket, 0, sizeof(tSendPacket));

  /* retrieve "ProtocollClass" and "CommunicationClass" from the DPM */
  usProtocollClass     = LE16_TO_HOST(HWIF_READ16(ptDevInst, ptSystemChannel->atChannelInfo[ptChannel->ulChannelNumber + COM_CHANNEL_OFFS].tCom.usProtocolClass));
  usCommunicationClass = LE16_TO_HOST(HWIF_READ16(ptDevInst, ptSystemChannel->atChannelInfo[ptChannel->ulChannelNumber + COM_CHANNEL_OFFS].tCom.usCommunicationClass));

   /* Device type   |  usProtocollClass          | usCommunicationClass
   -----------------+----------------------------+--------------------------
    PROFINET Device | HIL_PROT_CLASS_PROFINET_IO | HIL_COMM_CLASS_IO_DEVICE
    PROFIBUS Slave  | HIL_PROT_CLASS_PROFIBUS_DP | HIL_COMM_CLASS_SLAVE
    ETHERCAT Slave  | HIL_PROT_CLASS_ETHERCAT    | HIL_COMM_CLASS_SLAVE
    POWERLINK Node  | HIL_PROT_CLASS_POWERLINK   | HIL_COMM_CLASS_CONTROLLED_NODE
  */

  /* TODO: See protocol specific API manual or examples to setup warmstart/configuration packet(s)
           and send it */
  //if( CIFX_NO_ERROR == lRet)
  //{
    /*------------------------------------------------------------------------*/
    /* Process "Warmstart" will configure the communication channel...        */
    /*------------------------------------------------------------------------*/
    //lRet = DEV_ProcessWarmstart( ptChannel, &tSendPacket);
  //}

  return lRet;
}

/*****************************************************************************/
/*! I/O Demo (read inputs and write back to outputs)
/ * \param ptDevInstance Device instance
/ * \return CIFX_NO_ERROR on success                                          */
/*****************************************************************************/
int32_t IODemo(PCHANNELINSTANCE ptChannel)
{
  uint8_t     abData[SIZE_OF_IODATA];
  int32_t     lRet                    = CIFX_NO_ERROR;
  uint8_t     fHandshakeMode          = 0;
  PIOINSTANCE ptCommIn                = NULL;
  PIOINSTANCE ptCommOut               = NULL;

  OS_Memset( abData, 0, sizeof(abData));

  /* Execute a I/O data transfer */
  /* Initialize pointer to the Input and Output process data image from the hardware */
  ptCommIn  = ptChannel->pptIOInputAreas[0];
  ptCommOut = ptChannel->pptIOOutputAreas[0];

  /*------------------------------*/
  /* Handle Input data area       */
  /*------------------------------*/
  /* Read the actual I/O data handshake mode for the input areas */
  fHandshakeMode = DEV_GetIOBitstate(ptChannel, ptCommIn, 0);

  /* Check handshake mode mode */
  if(HIL_FLAGS_NONE == fHandshakeMode)
  {

    /* This is HIL_IO_MODE_UNCONTROLLED, copy data without using the handshake flags */

    HWIF_READN( ptChannel->pvDeviceInstance, abData, ptCommIn->pbDPMAreaStart, sizeof(abData));

  } else
  {

    /* This is a "controlled" mode, synchronized via the handshake flags */
    /* Check if the handshake flags are in the correct state, otherwise wait */
    /* until the state is reached and it is allowed to access the INPUT process data image */

    if(!DEV_WaitForBitState( ptChannel, ptCommIn->bHandshakeBit, fHandshakeMode, 100))
    {
      /* Failed to get the correct handshake flag state */
      lRet = CIFX_DEV_EXCHANGE_FAILED;
    } else
    {
      /* Read access is allowed, copy data to user buffer */
      HWIF_READN( ptChannel->pvDeviceInstance, abData, ptCommIn->pbDPMAreaStart, sizeof(abData));

      /* Signal / Acknowledge -> "Read Data Done" */
      DEV_ToggleBit( ptChannel, 1UL << ptCommIn->bHandshakeBit);
    }
  }

  /*------------------------------*/
  /* Handle Output data area      */
  /*------------------------------*/

  fHandshakeMode = DEV_GetIOBitstate(ptChannel, ptCommOut, 1);

  /* Check transfer mode */
  if(HIL_FLAGS_NONE == fHandshakeMode)
  {

    /* This is HIL_IO_MODE_UNCONTROLLED, copy data without using the handshake flags */
    HWIF_WRITEN( ptChannel->pvDeviceInstance, ptCommOut->pbDPMAreaStart, (void*)abData, sizeof(abData));

  } else
  {

    /* This is a "controlled" mode, synchronized via the handshake flags */
    /* Check if the handshake flags are in the correct state, otherwise wait */
    /* until the state is reached and it is allowed to access the OUTPUT process data image */

    if(!DEV_WaitForBitState( ptChannel, ptCommOut->bHandshakeBit, fHandshakeMode, 100))
    {
      lRet = CIFX_DEV_EXCHANGE_FAILED;
    } else
    {
      /* Write data */
      HWIF_WRITEN( ptChannel->pvDeviceInstance, ptCommOut->pbDPMAreaStart, (void*)abData, sizeof(abData));

      /* Signal -> "Write Data Done" */
      DEV_ToggleBit( ptChannel, 1UL << ptCommOut->bHandshakeBit);
    }
  }

  /* Return information about communication state */
  (void)DEV_IsCommunicating( ptChannel, &lRet);

  return lRet;
}


/*****************************************************************************/
/*! Packet Demo (read / write acyclic packets)
/ * \param ptChannel Channel handle
/ * \return CIFX_NO_ERROR on success                                         */
/*****************************************************************************/
void PacketDemo(PCHANNELINSTANCE ptChannel)
{
  int32_t   lRet            = CIFX_NO_ERROR;
  uint32_t  ulReceivePktCnt = 0;
  uint32_t  ulSendPktCnt    = 0;

  CIFX_PACKET tRecvPkt;
  OS_Memset( &tRecvPkt, 0, sizeof(tRecvPkt));

  /*------------------------------*/
  /* Read the mailbox state       */
  /*------------------------------*/
  lRet = DEV_GetMBXState( ptChannel, &ulReceivePktCnt, &ulSendPktCnt);
  if (CIFX_NO_ERROR != lRet)
    /* Unable to query the packet mailbox states */
    return;

  /*------------------------------*/
  /* Process receive packets      */
  /*------------------------------*/
  while  (ulReceivePktCnt > 0)
  {
    /* Try to read a packet from the mailbox */
    lRet = DEV_GetPacket( ptChannel, &tRecvPkt, sizeof(tRecvPkt), 0);

    if( CIFX_NO_ERROR != lRet)
    {
      /* Error reading a packet */

      break;
    }

    /* Check if we have a command or answer packet */
    if( 0 == (LE32_TO_HOST(tRecvPkt.tHeader.ulCmd) & 1))
    {
      /*--------------------------------------*/
      /* this is a command / indication packet*/
      /*--------------------------------------*/

      /* TODO: handle the indication packet */

    } else
    {
      /*-------------------------------------*/
      /* this is a answer / response packet  */
      /*-------------------------------------*/
      /* we have a packet, check for packet error */
      if( 0 == LE32_TO_HOST(tRecvPkt.tHeader.ulState))
      {
        /* Answer with error state */

        /* TODO: handle response packet */

      } else
      {
        /* Answer without error state */

        /* TODO: handle response packet */

      }
    }
    ulReceivePktCnt--;

  } /* end while */

  /*------------------------------*/
  /* Process send packets         */
  /*------------------------------*/

  if (ulSendPktCnt > 0)
  {
    /* We can send a packet, and the card is able to accept it */

    /* TODO: handle your request packets */

  }
}


/*****************************************************************************/
/*! Initialize the necessary data structures
* \param pbDPM           Pointer to the start of the netX Dual Port Memory
* \param ulDPMSize       Size of the netX Memory in bytes
* \param ptDevInstance   Pointer to a device instance structure
* \param ulStartTimeout  Wait time until hardware should be up and running
* \return DEV_NO_ERROR on success                                            */
/*****************************************************************************/
unsigned long cifXTkHWFunctions_InitializeDataStructures( uint8_t* pbDPM, uint32_t ulDPMSize, PDEVICEINSTANCE ptDevInstance, unsigned long ulStartTimeout)
{
  /* Check if firmware is READY because we need the DPM Layout */
  typedef struct SYSTEM_STARTUP_LAYOUTtag
  {
    HIL_DPM_SYSTEM_CHANNEL_T  tSystemChannel;
    HIL_DPM_HANDSHAKE_ARRAY_T tHskChannel;
  } SYSTEM_STARTUP_LAYOUT;

  SYSTEM_STARTUP_LAYOUT*  ptSysStartup = NULL;
  unsigned long           lRet         = DEV_NO_ERROR;
  uint32_t ulDiffTime   = 0L;
  int32_t  lStartTime   = 0L;

  /* Clean up and initialize the DEVICEINSTANCE structure */
  OS_Memset(ptDevInstance, 0, sizeof(*ptDevInstance));
  ptDevInstance->pbDPM     = pbDPM;
  ptDevInstance->ulDPMSize = ulDPMSize;

  /* Set pointer to handshake channel*/
  ptSysStartup = (SYSTEM_STARTUP_LAYOUT*)pbDPM;

  /*=====================================================================================================*/
  /* Now we have to wait until the System-Channel is ready, because we need a running system channel for */
  /* the further card handling                                                                           */
  /*=====================================================================================================*/
  /* Wait until System-Channel is READY */
  lStartTime   = (int32_t)OS_GetMilliSecCounter();

  /* This is the system channel which will reset the whole card */
  do
  {
    uint32_t ulHskFlags;
    ulHskFlags = LE32_TO_HOST(HWIF_READ32( ptDevInstance, ptSysStartup->tHskChannel.atHsk[0].ulValue));

    /* Check if DPM content is valid */
    if( (ulHskFlags == CIFX_DPM_INVALID_CONTENT)     ||
        (ulHskFlags == CIFX_DPM_NO_MEMORY_ASSIGNED)  ||
        (0 == (HWIF_READ8( ptDevInstance, ptSysStartup->tHskChannel.atHsk[0].t8Bit.bNetxFlags) & NSF_READY)))
    {
      /* Card is NOT READY */
      lRet = ERR_DEV_SYS_READY;
    } else
    {
      /* Card is READY */
      lRet = DEV_NO_ERROR;
      break;
    }
    /* Check time */
    ulDiffTime = OS_GetMilliSecCounter() - lStartTime;

    OS_Sleep(1);

  } while (ulDiffTime < ulStartTimeout);

  if( DEV_NO_ERROR == lRet)
  {
    /*===========================================================================*/
    /* Check the DPM "cookie" and find out if a firmware is running on the board */
    /* At least "BOOT" or "netX" must be available in the cookie                 */
    /*===========================================================================*/
    uint8_t   abCookie[4];
    HWIF_READN( ptDevInstance, (void*)abCookie, pbDPM, sizeof(abCookie));

    if( 0 == OS_Memcmp( abCookie, CIFX_DPMSIGNATURE_BSL_STR, sizeof(abCookie)))
    {
      /* Just the 2nd Stage Bootloader is active !!! */
      /* Card needs a Firmware for correct functionality */
      lRet = ERR_DEV_FW_MISSING;

    } else if( 0 == OS_Memcmp( abCookie, CIFX_DPMSIGNATURE_FW_STR, sizeof(abCookie)))
    {

      /*===========================================================================*/
      /* Initialize all necessary data structures for the DEV function handling    */
      /*===========================================================================*/
      if (CIFX_NO_ERROR != ((lRet = DEV_Initialize(ptDevInstance, DEV_CHANNEL_COUNT))))
      {
        /* Device initialization failed! */
        lRet = ERR_DEV_INIT;
      }

    } else
    {
      /* DPM identification failed! No valid Cookie! cifX/comX card is not present */
      lRet = ERR_DEV_DPM_IDENT;
    }
  }
  return lRet;
}


/*****************************************************************************/
/*! Uninitialize the data structures
*                                                                            */
/*****************************************************************************/
void cifXTkHWFunctions_UninitializeDataStructures( PDEVICEINSTANCE ptDevInstance)
{
  /* Uinitialize all data structures for the DEV function handling */
  DEV_Uninitialize( ptDevInstance);
}

/*****************************************************************************/
/*! Configure the selected device / communication channel
*
*   \return true on success                                                  */
/*****************************************************************************/
long cifXTkHWFunctions_ConfigureDevice( PCHANNELINSTANCE ptChannel, PCHANNELINSTANCE ptSysdevice)
{
  long lRet = DEV_NO_ERROR;

  /*----------------------------------------------------------------*/
  /* System device an communication channel must be ready           */
  /* Now check if the communication channel must be configured or   */
  /* if it us already "RUNNING"                                     */
  /*----------------------------------------------------------------*/

  /* if the running flag is set the channel is already configured,*/
  if ( DEV_IsRunning(ptChannel))
  {

    /* TODO: Decide if the communication channel should be configured   */
    /*       on each start.                                             */
    /*       If the device should be reconfigured, an already           */
    /*       existing configuration must be deleted first!              */
    /*       Consult the protocol specific examples on how to           */
    /*       initialize.                                                */

    /*------------------------------------------------------------------------*/
    /* Communication channel is RUNNING, which means it is configured         */
    /* To re-configure the channel, the existing configuration must be        */
    /* deleted first                                                          */
    /*------------------------------------------------------------------------*/
    /* Delete configuration files except the firmware file as          */
    /* cifX devices (RAM based CIFX50, netJack, ...) need to leave NXF */
    /*
    (void)DEV_RemoveChannelFiles(ptSysdevice, ptChannel->ulChannelNumber, DEV_TransferPacket, NULL, NULL, ".NXF");
    */
    /* Alternatively, only remove configuration files e.g. config.nxd, nwid.nxd etc. */

    /* If the device was configure using packet configuration, a delete configuration request needs to be sent to device */
    /*
    lRet = DEV_DeleteConfig( ptChannel);
    if( DEV_NO_ERROR != lRet)
      lRet = ERR_DEV_CONFIG;
    */

    /* At last, configure the communication channel again */
    /*
    lRet = ProcessDeviceConfiguration( ptChannel);
    if( CIFX_NO_ERROR != lRet)
      lRet = ERR_DEV_CONFIG;
    */

  }else
  {
    /*------------------------------------------------------------------------*/
    /* Communication channel is not RUNNING, which means it is not configured */
    /* Try to configure the channel by using "Warmstart" parameters.          */
    /*                                                                        */
    /*------------------------------------------------------------------------*/
    lRet = ProcessDeviceConfiguration( ptChannel);
    if( CIFX_NO_ERROR != lRet)
      lRet = ERR_DEV_CONFIG;
  }

  return lRet;
}

/*****************************************************************************/
/*! Wait until communicating
*
*   \return true on success                                                  */
/*****************************************************************************/
long cifXTkHWFunctions_WaitUntilCommunicating( PCHANNELINSTANCE ptChannel)
{
  unsigned long ulConfigCnt = 100;
  int32_t       lError      = 0;
  int32_t       lRet        = DEV_NO_ERROR;

  if( CIFX_NO_ERROR == lError)
  {
    /*------------------------------------------------------------------------*/
    /* After configuration, the channel is initialized                        */
    /* Usually "start of communication" takes a few seconds,                  */
    /* so we will wait 10 seconds until communication is running              */
    /*------------------------------------------------------------------------*/
    while ( ulConfigCnt--)
    {
      if ( DEV_IsCommunicating( ptChannel, &lError))
      {
        /* DEV_IsCommunicating() succeeded communication should be available */
        break;
      }

      OS_Sleep(100);              /* Wait a bit */
    }
  }

  if (CIFX_NO_ERROR != lError)
    /* We do not reach the communication state */
    lRet = ERR_DEV_COM;

  return lRet;
}


/*****************************************************************************/
/*! Hardware function example
 * \return 0 on success                                                      */
/*****************************************************************************/
int32_t cifXHWSample( void)
{
  int32_t  lDemoRet  = DEV_NO_ERROR;

  uint8_t*          pbDPM         = NULL;       /* This pointer must be loaded to the DPM address */
  uint32_t          ulDPMSize     = 0;          /* Size of the DPM in bytes */
  DEVICEINSTANCE    tDevInstance;               /* Global device data structure used by all DEV_xxx functions */

  /* Get pointer to the hardware dual-port memory and check if it is available */
  if ( FALSE == cifXTkHWFunctions_GetDPMPointer( &pbDPM, &ulDPMSize))
    /* Failed to get the hardware DPM pointer and size */
    return -1;

  /* Initialize the necessary data structures */
  if ( DEV_NO_ERROR == cifXTkHWFunctions_InitializeDataStructures( pbDPM, ulDPMSize, &tDevInstance, 10000))
  {
    /*----------------------------------------------------------------*/
    /* Read actual device states                                      */
    /*----------------------------------------------------------------*/
    PCHANNELINSTANCE ptSystemDevice = &tDevInstance.tSystemDevice;
    PCHANNELINSTANCE ptChannel      = tDevInstance.pptCommChannels[COM_CHANNEL];

#ifdef CIFX_TOOLKIT_HWIF
    /* realizes serialDPM read/write access to the netX DPM interface */
    if ( SERDPM_UNKNOWN == SerialDPM_Init( &tDevInstance))
    {
      /* failed to initialize serial DPM functions */
    }
#endif

    /* Wait for State acknowledge by the firmware */
    OS_Sleep(100);              /* Wait a bit */

    /* read the host flags of the system device, first time to synchronize our internal status */
    DEV_ReadHostFlags( ptSystemDevice, 0);

    /* read the host flags of the communication channel, first time to synchronise our internal status */
    DEV_ReadHostFlags( ptChannel, 0);

    /* check if "system device" is ready... */
    if (!DEV_IsReady( ptSystemDevice))
    {
      /* System device is not ready! */
      lDemoRet = ERR_DEV_SYS_READY;

    /* check if "communication channel" is ready... */
    } else if ( !DEV_IsReady(ptChannel))
    {
      /* Communication channel is not ready! */
      lDemoRet = ERR_DEV_COM_READY;

    } else
    {
      int32_t  lRet = CIFX_NO_ERROR;
      /*------------------------------------------------------------------------*/
      /* At this point we should have a running device and a configured         */
      /* communication channel.                                                 */
      /* Proceed with "NORMAL Stack Handling!                                   */
      /*------------------------------------------------------------------------*/
      /* Signal Host application is available */
      lRet = DEV_SetHostState( ptChannel, CIFX_HOST_STATE_READY, 1000);

      /* Configure the device */
      lDemoRet = cifXTkHWFunctions_ConfigureDevice( ptChannel, ptSystemDevice);
      //if( DEV_NO_ERROR != lDemoRet)
      //  printf("Error");

      /* Initialize and activate interrupt if configured */
      DEV_InitializeInterrupt ( &tDevInstance);

      if (DEV_NO_ERROR == lDemoRet)
      {
        /*------------------------------------------------------------------------*/
        /* At this point we should have a running device and a configured         */
        /* communication channel if no error is shown                             */
        /*------------------------------------------------------------------------*/
        uint32_t ulState = 0;

        /* Signal Host application is available */
        lRet = DEV_SetHostState( ptChannel, CIFX_HOST_STATE_READY, 1000);

        /* Switch ON the BUS communication */
        lRet = DEV_BusState( ptChannel, CIFX_BUS_STATE_ON, &ulState, 3000);

        /* TODO: Decide to wait until communication is available or just go to */
        /*       to the cyclic data handling and check the state there         */
        /* Wait for communication is available or do this during the cyclic program handling*/
        lDemoRet = cifXTkHWFunctions_WaitUntilCommunicating( ptChannel);

        /*------------------------------------------------------------------------*/
        if (lDemoRet == DEV_NO_ERROR)
        {
          /* device is "READY", "RUNNING" and "COMMUNICATING" */
          /* Start cyclic demo with I/O Data-Transfer and packet data transfer */
          unsigned long ulCycCnt = 0;
          //uint32_t ulTriggerCount = 0;

          /* Cyclic I/O and packet handling for 'ulCycCnt'times */
          while( ulCycCnt < DEMO_CYCLES)
          {
            /* Start and trigger watchdog function, if necessary */
            //DEV_TriggerWatchdog(ptChannel, CIFX_WATCHDOG_START, &ulTriggerCount);

            /* Handle I/O data transfer */
            (void)IODemo(ptChannel);

            /* Handle packet transfer */
            PacketDemo ( ptChannel);

            ulCycCnt++;
          }

          /* Stop watchdog function, if it was previously started */
          //DEV_TriggerWatchdog(ptChannel, CIFX_WATCHDOG_STOP, &ulTriggerCount);
        }

        /* Switch OFF the BUS communication / dot't wait */
        lRet = DEV_BusState( ptChannel, CIFX_BUS_STATE_OFF, &ulState, 0);

        /* Signal Host application is not available anymore / don't wait */
        lRet = DEV_SetHostState( ptChannel, CIFX_HOST_STATE_NOT_READY, 0);
      }

      /* Uninitialize interrupt */
      DEV_UninitializeInterrupt ( &tDevInstance);
    } /*lint !e438 !e550 */ /* 'lRet' not used */
  }

  /* Clean up all used memory areas and pointers */
  cifXTkHWFunctions_UninitializeDataStructures( &tDevInstance);

  /* cifXTkHWFunctions clean up */
  cifXTkHWFunctions_FreeDPMPointer( pbDPM);

  return lDemoRet;
}
